(() => {
	const modal = document.getElementById('colorModal');
	const colorSwatches = modal.querySelectorAll('.color-swatch');
	const removeBtn = document.getElementById('removeHighlightBtn');
	const closeModalBtn = document.getElementById('closeModalBtn');
		
	const content = document.querySelector('.content_container');
		
	const STORAGE_KEY = 'textHighlights_' + window.location.pathname;

	let selectedRange = null;
	let selectedHighlight = null;
	let processedHTML = '';	

	let openedByHighlightClick = false;

	function getContrastColor(hex) {
		hex = hex.replace("#", "");
		const r = parseInt(hex.substr(0, 2), 16);
		const g = parseInt(hex.substr(2, 2), 16);
		const b = parseInt(hex.substr(4, 2), 16);
		const yiq = (r * 299 + g * 587 + b * 114) / 1000;
		return yiq >= 128 ? "#000" : "#fff";
	}

	function loadHighlights() {
		const json = localStorage.getItem(STORAGE_KEY);
		if (!json) return [];
		try {
			return JSON.parse(json);
		} catch {
			return [];
		}
	}

	function saveHighlights(data) {
		localStorage.setItem(STORAGE_KEY, JSON.stringify(data));
	}
	
	function getGlobalOffsets(range) {
	if (!content) return { startOffset: 0, endOffset: 0 };

		const preSelectionRange = range.cloneRange();
		preSelectionRange.selectNodeContents(content);	
		preSelectionRange.setEnd(range.startContainer, range.startOffset);
		const startOffset = preSelectionRange.toString().length;
		const endOffset = startOffset + range.toString().length;
		return { startOffset, endOffset };
	}
		
	function findTextNode(container, offset) {
	const walker = document.createTreeWalker(container, NodeFilter.SHOW_TEXT, null, false);
	let currentOffset = 0;
		let node;
		while (node = walker.nextNode()) {
			const nextOffset = currentOffset + node.length;
			if (offset >= currentOffset && offset <= nextOffset) {
				return { node, offset: offset - currentOffset };
			}
			currentOffset = nextOffset;
		}
		return null;
	}

	function findFirstMeaningfulTextNode(element) {
		for (const child of element.childNodes) {
			if (child.nodeType === Node.TEXT_NODE) {
				if (child.nodeValue.trim().length > 0) {
					return child;	
				}
			} else if (child.nodeType === Node.ELEMENT_NODE) {
				if (['IMG', 'BR', 'HR', 'INPUT', 'BUTTON', 'TEXTAREA', 'SELECT'].includes(child.tagName)) {
						continue;
				}
				const found = findFirstMeaningfulTextNode(child);
				if (found) {
					return found;
				}
			}
		}
		return null;	
	}

	function preprocessContentWithLeadingSpaces() {
		if (!content) {
				console.error("Fehler: Das Element mit der Klasse 'content_container' wurde nicht gefunden. Vorverarbeitung abgebrochen.");
				return;	
		}
		const tempDiv = document.createElement('div');
		tempDiv.innerHTML = content.innerHTML;	
		const containerElements = tempDiv.querySelectorAll('p, h1, h2, h3, h4, h5, h6, li, div');	
		containerElements.forEach(container => {
			const firstTextNode = findFirstMeaningfulTextNode(container);
			if (firstTextNode) {
				if (!firstTextNode.nodeValue.startsWith(' ')) {
					firstTextNode.nodeValue = ' ' + firstTextNode.nodeValue;
				}
			}
		});
		processedHTML = tempDiv.innerHTML;	
	}


	function renderAllHighlights() {
		if (!content) {
				console.error("Fehler: Das Element mit der Klasse 'content_container' wurde nicht gefunden. Highlights können nicht gerendert werden.");
				return;
		}
		const highlights = loadHighlights();
		content.innerHTML = processedHTML;
		if (!highlights.length) return;
		highlights.sort((a, b) => b.endOffset - a.endOffset);
		highlights.forEach((h) => {
			const startNode = findTextNode(content, h.startOffset);
			const endNode = findTextNode(content, h.endOffset);
			if (!startNode || !endNode) {
				console.warn('Highlight-Position nicht gefunden, wird übersprungen:', h);
				return;
			}
			const range = document.createRange();
			range.setStart(startNode.node, startNode.offset);
			range.setEnd(endNode.node, endNode.offset);
			const span = document.createElement('span');
			span.className = 'highlight';
			span.setAttribute('data-id', h.id);
			span.style.backgroundColor = h.color;
			span.style.color = getContrastColor(h.color) + ' !important';

				try {
				range.surroundContents(span);
				} catch (e) {
				console.error("Fehler beim Umschließen der Markierung. Range:", range, e);
				const highlightedText = range.extractContents();
				span.appendChild(highlightedText);
				range.insertNode(span);
			}
		});
		}

		function isSelectionWithinSingleBlock(range) {
		if (range.startContainer === range.endContainer) {
		return true;
		}
		const startParent = range.startContainer.nodeType === Node.ELEMENT_NODE ? range.startContainer : range.startContainer.parentNode;
		const endParent = range.endContainer.nodeType === Node.ELEMENT_NODE ? range.endContainer : range.endContainer.parentNode;
		let currentStartBlock = startParent;
		while(currentStartBlock && currentStartBlock !== content && !currentStartBlock.matches('p, li, div, h1, h2, h3, h4, h5, h6')) {
				currentStartBlock = currentStartBlock.parentNode;
		}
		let currentEndBlock = endParent;
		while(currentEndBlock && currentEndBlock !== content && !currentEndBlock.matches('p, li, div, h1, h2, h3, h4, h5, h6')) {
				currentEndBlock = currentEndBlock.parentNode;
		}
		if (currentStartBlock && currentEndBlock && currentStartBlock !== currentEndBlock) {
				return false;
		}
		let ancestor = range.commonAncestorContainer;
		if (ancestor.nodeType === Node.TEXT_NODE) {
				ancestor = ancestor.parentNode;
		}
		if (ancestor === content) {
				let startRootChild = range.startContainer;
				while(startRootChild && startRootChild.parentNode !== content && startRootChild !== content) {
						startRootChild = startRootChild.parentNode;
				}
				let endRootChild = range.endContainer;
				while(endRootChild && endRootChild.parentNode !== content && endRootChild !== content) {
						endRootChild = endRootChild.parentNode;
				}
				if (startRootChild && endRootChild && startRootChild !== endRootChild) {
						return false;
				}
		}
		return true;
	}

	function openModalAtFixedLocation() {
			modal.style.display = "block";
			modal.style.left = '0px';
			modal.style.top = '0px';
			removeBtn.style.display = (selectedHighlight !== null) ? "inline-block" : "none";
			}

	function deleteHighlight(id) {
		if (!id) {
				console.warn("deleteHighlight called without an ID.");
				return;
		}
		const highlights = loadHighlights();
		const filtered = highlights.filter((h) => h.id !== id);
		saveHighlights(filtered);
		renderAllHighlights();
		selectedHighlight = null;
		removeBtn.style.display = "none";
	}

	// --- Event Listeners ---

	colorSwatches.forEach((swatch) => {
		swatch.addEventListener("click", () => {
			const newColor = swatch.getAttribute("data-color");
			const highlights = loadHighlights();
			if (selectedHighlight) {
				const id = selectedHighlight.getAttribute("data-id");
				const highlightIndex = highlights.findIndex(h => h.id === id);
				if (highlightIndex !== -1) {
					highlights[highlightIndex].color = newColor;
					saveHighlights(highlights);
					renderAllHighlights();
				}
			} else if (selectedRange) {
				const { startOffset, endOffset } = getGlobalOffsets(selectedRange);
				const id = "id-" + Date.now() + "-" + Math.floor(Math.random() * 1000);
				const selectedText = selectedRange.toString();
				highlights.push({
					id,
					color: newColor,
					startOffset,
					endOffset,
					text: selectedText
				});
				saveHighlights(highlights);
				renderAllHighlights();
			}
			modal.style.display = "none";
			window.getSelection().removeAllRanges();
			selectedRange = null;
			selectedHighlight = null;
			openedByHighlightClick = false;
		});
	});

	content.addEventListener("click", (e) => {
		if (!content) return;
		const target = e.target;
		if (target.classList.contains("highlight")) {
			selectedHighlight = target;
			selectedRange = null;
			openedByHighlightClick = true;
			openModalAtFixedLocation();
		} else {
			if (!document.getSelection().toString()) {
				modal.style.display = "none";
				removeBtn.style.display = "none";
				selectedHighlight = null;
				openedByHighlightClick = false;
			}
		}
	});

	removeBtn.addEventListener("click", () => {
		if (!selectedHighlight) {
				console.warn("No highlight element selected to remove.");
				return;
		}
		const id = selectedHighlight.getAttribute("data-id");
		if (id) {
				deleteHighlight(id);
		} else {
				console.error("Selected highlight has no data-id attribute.");
		}
	modal.style.display = "none";
		window.getSelection().removeAllRanges();
		openedByHighlightClick = false;
	});

	closeModalBtn.addEventListener("click", () => {
		modal.style.display = "none";
		removeBtn.style.display = "none";
		selectedHighlight = null;
		window.getSelection().removeAllRanges();
		openedByHighlightClick = false;
	});

	document.addEventListener("mouseup", () => {
		setTimeout(() => {
			if (!content) return;
			const selection = window.getSelection();
			const isSelectionActive = selection.rangeCount > 0 && selection.toString().trim() !== "";
			if (isSelectionActive) {
				selectedHighlight = null;
				selectedRange = selection.getRangeAt(0);
				if (!isSelectionWithinSingleBlock(selectedRange)) {
					alert("Bitte markieren Sie nur einen Textbereich innerhalb eines einzelnen Absatzes oder Listenelements.");
					selection.removeAllRanges();
					modal.style.display = "none";
					return;
				}
				openedByHighlightClick = false;
				openModalAtFixedLocation();
			} else {
				if (!openedByHighlightClick && modal.style.display === "block") {
					modal.style.display = "none";
					removeBtn.style.display = "none";
					selectedHighlight = null;
					}
				openedByHighlightClick = false;
			}
		}, 10);
	});

	window.addEventListener("load", () => {
		preprocessContentWithLeadingSpaces();
		renderAllHighlights();
	});
})();