<!--
## Lizenz
Dieses Tool wird unter der MIT-Lizenz bereitgestellt.  

Sie dürfen:
- das Skript frei verwenden, kopieren und weitergeben,
- es für private und kommerzielle Projekte einsetzen,
- Modifikationen vornehmen und eigene Versionen erstellen.

Bedingungen:
- In allen Kopien oder wesentlichen Teilen des Skripts muss der obige Copyright-Hinweis 
  sowie dieser Lizenztext enthalten bleiben.  
- Es wird keinerlei Haftung oder Garantie übernommen. Nutzung auf eigenes Risiko.

Haftungsausschluss:
Dieses Tool wird ohne jede Gewährleistung bereitgestellt. Die Nutzung erfolgt vollständig auf eigenes Risiko.
-->

<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Advanced Color Palette Generator</title>
    <link rel="stylesheet" href="styles.css">
</head>
<body>
    <div id="container">
	

	
        <h1>Advanced Color Palette Generator</h1>
        <div id="controls">
            <div>
                <label for="startColor">Startfarbe:</label>
                <input type="color" id="startColor" class="color-picker">
            </div>
            <div>
                <label for="endColor">Endfarbe:</label>
                <input type="color" id="endColor" class="color-picker">
            </div>
            <div class="slider-container">
                <label for="intensity">Intensität:</label>
                <input type="range" id="intensity" class="slider" min="0" max="100" value="50">
            </div>
            <div class="slider-container">
                <label for="saturation">Sättigung:</label>
                <input type="range" id="saturation" class="slider" min="0" max="100" value="100">
            </div>
            <div class="slider-container">
                <label for="hueShift">Farbtonverschiebung:</label>
                <input type="range" id="hueShift" class="slider" min="0" max="360" value="0">
            </div>
			<div class="slider-container">
				<label for="analogRange">Analoger Bereich:</label>
				<input type="range" id="analogRange" class="slider" min="0" max="120" value="30">
			</div>

            <div>
                <label for="mode">Modus:</label>
                <select id="mode">
                    <option value="linear">Linear</option>
                    <option value="radial">Radial</option>
                    <option value="complementary">Komplementär</option>
                    <option value="triadic">Triadisch</option>
                    <option value="tetradic">Tetradisch</option>
                    <option value="analogous">Analog</option>
                    <option value="monochromatic">Monochrom</option>
                </select>
            </div>
            <div>
                <label for="colorCount">Anzahl der Farben:</label>
                <input type="number" id="colorCount" min="2" max="60" value="20">
            </div>
            <div>
                <label for="colorSpace">Farbraum:</label>
                <select id="colorSpace">
                    <option value="hex">HEX</option>
					<option value="rgb">RGB</option>
                    <option value="hsl">HSL</option>
                    <option value="lab">LAB</option>
                </select>
            </div>
            <!-- <button id="generate">Generieren</button> -->
            <button class="randomize" id="randomize">Zufällige Palette</button>
            <button class="invertPalette" id="invertPalette">Palette invertieren</button>
        </div>
        <div id="palette"></div>
		
		<div id="gradientEditorContainer">
  <h3>Visueller Gradient-Editor</h3>
  
  <div id="preview_grid">
	<div id="gradientPreview"></div>
  </div>
  <div id="gradientStopsEditor"></div>
  
  <div id="colorPickerModeOption">
    <label>
      <input type="checkbox" id="colorPickerMode">
      Farbpicker-Modus aktivieren
    </label>
  </div>

  <div id="gradientDirections">
    <h4>Gradient-Richtung:</h4>
    <div class="direction-grid">
      <div class="direction-column">
        <label>
  <input type="radio" name="gradientDirection" value="to right" checked><span class="direction-arrow right">➜</span> (90°)</label>
        <label><input type="radio" name="gradientDirection" value="to left"><span class="direction-arrow left">➜</span> (270°)</label>
      </div>
      <div class="direction-column">
        <label><input type="radio" name="gradientDirection" value="to bottom"><span class="direction-arrow down">➜</span> (180°)</label>
        <label><input type="radio" name="gradientDirection" value="to top"><span class="direction-arrow up">➜</span> (0°)</label>
      </div>
      <div class="direction-column">
        <label><input type="radio" name="gradientDirection" value="to bottom right"><span class="direction-arrow bottom-right">➜</span>  (135°)</label>
        <label><input type="radio" name="gradientDirection" value="to bottom left"><span class="direction-arrow bottom-left">➜</span>  (225°)</label>
      </div>
      <div class="direction-column">
        <label><input type="radio" name="gradientDirection" value="to top right"><span class="direction-arrow top-right">➜</span> (45°)</label>
        <label><input type="radio" name="gradientDirection" value="to top left"><span class="direction-arrow top-left">➜</span> (315°)</label>
      </div>
      <div class="direction-column">
  <label>
    <input type="radio" name="gradientDirection" value="custom">
    <span class="direction-arrow custom">➜</span> Benutzerdefiniert
  </label>
  <input type="number" id="customAngle" min="0" max="360" value="90">
</div>
    </div>
  </div>
  <div id="transparencyOptions">
    <label>
      <input type="checkbox" id="useTransparency">
      8-stellige Hex-Werte verwenden (mit Transparenz) Deckfähigkeit 0% - 100%
    </label>
    <input type="number" id="transparencyValue" min="0" max="100" value="100">
  </div>

  <button class="updateGradient" id="updateGradient">Gradient aktualisieren</button>
  <button class="exportGradientCSS" id="exportGradientCSS">Gradient-CSS exportieren</button>
  <button class="toggleImportBtn" id="toggleImportBtn">Importieren</button>
  <button class="exportCSS" id="exportCSS">Als CSS exportieren</button>
  <button class="exportSVG" id="exportSVG">Als SVG exportieren</button>
  <button class="exportJSON" id="exportJSON">Als JSON exportieren</button>
</div>



<div id="importSection" style="display: none;">
  <div id="importer">
    <textarea class="importTextarea" id="importText" rows="5" placeholder="CSS-Code einfügen"></textarea>
    <button class="importButton" id="importButton">Importieren</button>
  </div>
</div>

<div id="copyMessage" style="display: none; position: fixed; top: 10px; right: 10px; padding: 10px; background-color: #4CAF50; color: white; border-radius: 5px;">CSS wurde in die Zwischenablage kopiert</div>

    </div>
    <script>
const startColorPicker = document.getElementById('startColor');
const endColorPicker = document.getElementById('endColor');
const intensitySlider = document.getElementById('intensity');
const saturationSlider = document.getElementById('saturation');
const hueShiftSlider = document.getElementById('hueShift');
const modeSelect = document.getElementById('mode');
const colorCountInput = document.getElementById('colorCount');
const colorSpaceSelect = document.getElementById('colorSpace');
// const generateButton = document.getElementById('generate');
const randomizeButton = document.getElementById('randomize');
const invertPaletteButton = document.getElementById('invertPalette');
const exportCSSButton = document.getElementById('exportCSS');
const exportSVGButton = document.getElementById('exportSVG');
const exportJSONButton = document.getElementById('exportJSON');
const paletteContainer = document.getElementById('palette');

startColorPicker.addEventListener('input', generatePalette);
endColorPicker.addEventListener('input', generatePalette);
intensitySlider.addEventListener('input', generatePalette);
saturationSlider.addEventListener('input', generatePalette);
hueShiftSlider.addEventListener('input', generatePalette);
modeSelect.addEventListener('change', generatePalette);
colorCountInput.addEventListener('input', generatePalette);
colorSpaceSelect.addEventListener('change', generatePalette);

function hexToRgb(hex) {
    const r = parseInt(hex.slice(1, 3), 16);
    const g = parseInt(hex.slice(3, 5), 16);
    const b = parseInt(hex.slice(5, 7), 16);
    return [r, g, b];
}

function rgbToHex(r, g, b) {
    return `#${((1 << 24) + (r << 16) + (g << 8) + b).toString(16).slice(1)}`;
}

function hslToRgb(h, s, l) {
    let r, g, b;
    if (s === 0) {
        r = g = b = l;
    } else {
        const hue2rgb = (p, q, t) => {
            if (t < 0) t += 1;
            if (t > 1) t -= 1;
            if (t < 1/6) return p + (q - p) * 6 * t;
            if (t < 1/2) return q;
            if (t < 2/3) return p + (q - p) * (2/3 - t) * 6;
            return p;
        };
        const q = l < 0.5 ? l * (1 + s) : l + s - l * s;
        const p = 2 * l - q;
        r = hue2rgb(p, q, h + 1/3);
        g = hue2rgb(p, q, h);
        b = hue2rgb(p, q, h - 1/3);
    }
    return [Math.round(r * 255), Math.round(g * 255), Math.round(b * 255)];
}

function rgbToHsl(r, g, b) {
    r /= 255, g /= 255, b /= 255;
    const max = Math.max(r, g, b), min = Math.min(r, g, b);
    let h, s, l = (max + min) / 2;
    if (max === min) {
        h = s = 0;
    } else {
        const d = max - min;
        s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
        switch (max) {
            case r: h = (g - b) / d + (g < b ? 6 : 0); break;
            case g: h = (b - r) / d + 2; break;
            case b: h = (r - g) / d + 4; break;
        }
        h /= 6;
    }
    return [h, s, l];
}
function generatePalette() {
    const startColor = hexToRgb(startColorPicker.value);
    const endColor = hexToRgb(endColorPicker.value);
    const intensity = intensitySlider.value / 100;
    const saturation = saturationSlider.value / 100;
    const hueShift = parseInt(hueShiftSlider.value);
    const mode = modeSelect.value;
    const colorCount = parseInt(colorCountInput.value);
    const colorSpace = colorSpaceSelect.value;

    let colors = [];

    switch (mode) {
        case 'linear':
            for (let i = 0; i < colorCount; i++) {
                const r = Math.round(startColor[0] + (endColor[0] - startColor[0]) * (i / (colorCount - 1)));
                const g = Math.round(startColor[1] + (endColor[1] - startColor[1]) * (i / (colorCount - 1)));
                const b = Math.round(startColor[2] + (endColor[2] - startColor[2]) * (i / (colorCount - 1)));
                colors.push(rgbToHex(r, g, b));
            }
            break;
        case 'radial':
            for (let i = 0; i < colorCount; i++) {
                const hue = (i / colorCount) * 360 + hueShift;
                const [r, g, b] = hslToRgb((hue % 360) / 360, saturation, intensity);
                colors.push(rgbToHex(r, g, b));
            }
            break;
        case 'complementary':
            const startHsl = rgbToHsl(...startColor);
            const endHsl = rgbToHsl(...endColor);
            for (let i = 0; i < colorCount; i++) {
                const hue = (startHsl[0] + (endHsl[0] - startHsl[0]) * (i / (colorCount - 1)) + hueShift / 360) % 1;
                const [r, g, b] = hslToRgb(hue, saturation, intensity);
                colors.push(rgbToHex(r, g, b));
            }
            break;
        case 'triadic':
            const baseHue = rgbToHsl(...startColor)[0];
            for (let i = 0; i < colorCount; i++) {
                const hue = (baseHue + (i / colorCount) * 0.33333 + hueShift / 360) % 1;
                const [r, g, b] = hslToRgb(hue, saturation, intensity);
                colors.push(rgbToHex(r, g, b));
            }
            break;
        case 'tetradic':
            const baseHue2 = rgbToHsl(...startColor)[0];
            for (let i = 0; i < colorCount; i++) {
                const hue = (baseHue2 + (i / colorCount) * 0.25 + hueShift / 360) % 1;
                const [r, g, b] = hslToRgb(hue, saturation, intensity);
                colors.push(rgbToHex(r, g, b));
            }
            break;
        case 'analogous':
    const baseHue3 = rgbToHsl(...startColor)[0];
    const analogRange = document.getElementById('analogRange').value / 360; // Umwandlung in HSL-Wert
    for (let i = 0; i < colorCount; i++) {
        const hue = (baseHue3 + (i / colorCount) * analogRange + hueShift / 360) % 1;
        const [r, g, b] = hslToRgb(hue, saturation, intensity);
        colors.push(rgbToHex(r, g, b));
    }
    break;
        case 'monochromatic':
            const baseHsl = rgbToHsl(...startColor);
            for (let i = 0; i < colorCount; i++) {
                const lightness = intensity * (1 - i / (colorCount - 1));
                const [r, g, b] = hslToRgb(baseHsl[0], saturation, lightness);
                colors.push(rgbToHex(r, g, b));
            }
            break;
    }

    if (colorSpace === 'hsl') {
        colors = colors.map(color => {
            const [r, g, b] = hexToRgb(color);
            const [h, s, l] = rgbToHsl(r, g, b);
            return `hsl(${Math.round(h * 360)}, ${Math.round(s * 100)}%, ${Math.round(l * 100)}%)`;
        });
    } else if (colorSpace === 'lab') {
        colors = colors.map(color => {
            const [r, g, b] = hexToRgb(color);
            const [l, a, b_] = rgbToLab(r, g, b);
            return `lab(${Math.round(l)}% ${Math.round(a)} ${Math.round(b_)})`;
        });
    }

    displayPalette(colors);
}

function displayPalette(colors) {
    paletteContainer.innerHTML = '';
    colors.forEach(color => {
        const colorBox = document.createElement('div');
        colorBox.className = 'color-box';
        colorBox.style.backgroundColor = color;

        const colorValue = document.createElement('span');
        colorValue.className = 'color-value';
        
        // Neue Logik für RGB
        if (colorSpaceSelect.value === 'rgb') {
            const [r, g, b] = hexToRgb(color);
            colorValue.textContent = `rgb(${r}, ${g}, ${b})`;
        } else {
            colorValue.textContent = color;
        }

        colorValue.addEventListener('click', () => {
            navigator.clipboard.writeText(colorValue.textContent);
            showNotification(`Farbe ${colorValue.textContent} in die Zwischenablage kopiert!`);
        });

        colorBox.appendChild(colorValue);
        paletteContainer.appendChild(colorBox);
    });
}

// Hilfsfunktion für die Umwandlung von HEX zu RGB
function hexToRgb(hex) {
    const r = parseInt(hex.slice(1, 3), 16);
    const g = parseInt(hex.slice(3, 5), 16);
    const b = parseInt(hex.slice(5, 7), 16);
    return [r, g, b];
}


function showNotification(message) {
    const notification = document.getElementById('notification');
    notification.textContent = message;
    notification.style.display = 'block';
    setTimeout(() => {
        notification.style.display = 'none';
    }, 6000);
}

function randomizePalette() {
    startColorPicker.value = `#${Math.floor(Math.random()*16777215).toString(16).padStart(6, '0')}`;
    endColorPicker.value = `#${Math.floor(Math.random()*16777215).toString(16).padStart(6, '0')}`;
    intensitySlider.value = Math.floor(Math.random() * 101);
    saturationSlider.value = Math.floor(Math.random() * 101);
    hueShiftSlider.value = Math.floor(Math.random() * 361);
    // Entfernt: modeSelect.selectedIndex = Math.floor(Math.random() * modeSelect.options.length);
    // Entfernt: colorCountInput.value = Math.floor(Math.random() * 46) + 5;
    // Entfernt: colorSpaceSelect.selectedIndex = Math.floor(Math.random() * colorSpaceSelect.options.length);
    generatePalette();
}




function invertPalette() {
    const colors = Array.from(paletteContainer.querySelectorAll('.color-box'))
        .map(box => box.style.backgroundColor);
    const invertedColors = colors.map(color => {
        const rgb = color.match(/\d+/g).map(Number);
        return rgbToHex(255 - rgb[0], 255 - rgb[1], 255 - rgb[2]);
    });
    displayPalette(invertedColors);
}

function exportCSS() {
    const colors = Array.from(paletteContainer.querySelectorAll('.color-value'))
        .map(span => span.textContent);
    let css = ':root {\n';
    colors.forEach((color, index) => {
        css += `  --color-${index + 1}: ${color};\n`;
    });
    css += '}\n\n';

    const stepPercent = 100 / (colors.length - 1);
    let gradientStops = colors.map((_, index) => {
        const percent = (stepPercent * index).toFixed(2);
        return `var(--color-${index + 1}) ${percent}%`;
    }).join(', ');

    css += `background: linear-gradient(90deg, ${gradientStops});\n`;

    navigator.clipboard.writeText(css);
    alert('CSS-Variablen und Gradient in die Zwischenablage kopiert!');
}

function exportSVG() {
    const colors = Array.from(paletteContainer.querySelectorAll('.color-value'))
        .map(span => span.textContent);
    const svgWidth = 500;
    const svgHeight = 50;
    const numColors = colors.length;
    
    let svg = `<svg xmlns="http://www.w3.org/2000/svg" width="${svgWidth}" height="${svgHeight}">\n<defs>\n`;
    svg += `<linearGradient id="grad" x1="0%" y1="0%" x2="100%" y2="0%">\n`;
    colors.forEach((color, index) => {
        const offsetPercent = (index / (numColors - 1)) * 100;
        svg += `  <stop offset="${offsetPercent}%" stop-color="${color}" />\n`;
    });
    svg += `</linearGradient>\n</defs>\n`;
    svg += `<rect width="${svgWidth}" height="${svgHeight}" fill="url(#grad)" />\n`;
    svg += `</svg>`;
    
    navigator.clipboard.writeText(svg);
    alert('SVG mit LinearGradient in die Zwischenablage kopiert!');
}


function exportJSON() {
    const colors = Array.from(paletteContainer.querySelectorAll('.color-value'))
        .map(span => span.textContent);
    const json = JSON.stringify(colors, null, 2);
    navigator.clipboard.writeText(json);
    alert('JSON in die Zwischenablage kopiert!');
}

const analogRangeSlider = document.getElementById('analogRange');
analogRangeSlider.addEventListener('input', generatePalette);


// Hilfsfunktion für LAB-Farbraum
function rgbToLab(r, g, b) {
    let x, y, z;
    r /= 255; g /= 255; b /= 255;
    r = (r > 0.04045) ? Math.pow((r + 0.055) / 1.055, 2.4) : r / 12.92;
    g = (g > 0.04045) ? Math.pow((g + 0.055) / 1.055, 2.4) : g / 12.92;
    b = (b > 0.04045) ? Math.pow((b + 0.055) / 1.055, 2.4) : b / 12.92;
    x = (r * 0.4124 + g * 0.3576 + b * 0.1805) / 0.95047;
    y = (r * 0.2126 + g * 0.7152 + b * 0.0722) / 1.00000;
    z = (r * 0.0193 + g * 0.1192 + b * 0.9505) / 1.08883;
    x = (x > 0.008856) ? Math.pow(x, 1/3) : (7.787 * x) + 16/116;
    y = (y > 0.008856) ? Math.pow(y, 1/3) : (7.787 * y) + 16/116;
    z = (z > 0.008856) ? Math.pow(z, 1/3) : (7.787 * z) + 16/116;
    return [(116 * y) - 16, 500 * (x - y), 200 * (y - z)];
}

// Event Listener
// generateButton.addEventListener('click', generatePalette);
randomizeButton.addEventListener('click', randomizePalette);
invertPaletteButton.addEventListener('click', invertPalette);
exportCSSButton.addEventListener('click', exportCSS);
exportSVGButton.addEventListener('click', exportSVG);
exportJSONButton.addEventListener('click', exportJSON);

// Initialer Aufruf
generatePalette();

// Visueller Editor

// Visueller Editor

document.addEventListener('DOMContentLoaded', function() {
    const gradientPreview = document.getElementById('gradientPreview');
    const gradientStopsEditor = document.getElementById('gradientStopsEditor');
    const updateGradientBtn = document.getElementById('updateGradient');
    const exportGradientCSSBtn = document.getElementById('exportGradientCSS');
    const customAngleInput = document.getElementById('customAngle');
    const useTransparencyCheckbox = document.getElementById('useTransparency');
    const transparencyInput = document.getElementById('transparencyValue');
    const colorPickerModeCheckbox = document.getElementById('colorPickerMode');
    let colorPickerMode = false;

    // Neue Elemente für den Import
    const importText = document.getElementById('importText');
    const importButton = document.getElementById('importButton');

    function initGradientEditor(colorStops = []) {
        gradientStopsEditor.innerHTML = '';

        if (colorStops.length === 0) {
            colorStops = Array.from(document.querySelectorAll('.color-value'))
                .map((span, index, array) => ({
                    color: span.textContent,
                    position: (index / (array.length - 1)) * 100
                }));
        }

        colorStops.forEach((stop) => {
            addColorStop(stop.color, stop.position);
        });

        updateGradientPreview();
    }

    function addColorStop(color, position) {
        const stopElement = document.createElement('div');
        stopElement.className = 'gradientStop';
        stopElement.style.backgroundColor = color;
        stopElement.style.left = `${position !== null ? position : 0}%`;
        
        stopElement.addEventListener('mousedown', handleStopInteraction);
        stopElement.addEventListener('contextmenu', handleRemoveStop);
        
        gradientStopsEditor.appendChild(stopElement);
    }

    function handleRemoveStop(e) {
        e.preventDefault();
        if (gradientStopsEditor.children.length > 2) {
            e.target.remove();
            updateGradientPreview();
        } else {
            alert('Minimum 2 colors required!');
        }
    }

    gradientStopsEditor.addEventListener('dblclick', function(e) {
        if (e.target === this) {
            const newPosition = (e.offsetX / this.offsetWidth) * 100;
            addColorStop('#ffffff', newPosition);
            updateGradientPreview();
        }
    });

    function handleStopInteraction(e) {
        if (colorPickerMode) {
            openColorPicker(e.target);
        } else {
            startDragging(e);
        }
    }

    function startDragging(e) {
        const stop = e.target;
        const startX = e.clientX;
        const startLeft = parseFloat(stop.style.left);
        const stops = Array.from(document.querySelectorAll('.gradientStop'));
        const index = stops.indexOf(stop);
        
        function onMouseMove(e) {
            const deltaX = e.clientX - startX;
            let newPosition = startLeft + (deltaX / gradientStopsEditor.offsetWidth) * 100;
            
            newPosition = Math.max(0, Math.min(100, newPosition));
            
            stop.style.left = `${newPosition}%`;
            
            stops.sort((a, b) => parseFloat(a.style.left) - parseFloat(b.style.left));
            
            stops.forEach(s => gradientStopsEditor.appendChild(s));
            
            updateGradientPreview();
        }

        function onMouseUp() {
            document.removeEventListener('mousemove', onMouseMove);
            document.removeEventListener('mouseup', onMouseUp);
        }

        document.addEventListener('mousemove', onMouseMove);
        document.addEventListener('mouseup', onMouseUp);
    }

    function openColorPicker(stop) {
        const input = document.createElement('input');
        input.type = 'color';
        input.value = rgbToHex(...stop.style.backgroundColor.match(/\d+/g).map(Number));
        input.addEventListener('input', (e) => {
            stop.style.backgroundColor = e.target.value;
            updateGradientPreview();
        });
        input.click();
    }

    function updateGradientPreview() {
        const stops = document.querySelectorAll('.gradientStop');
        const gradientStops = Array.from(stops).map(stop => {
            const rgb = stop.style.backgroundColor.match(/\d+/g);
            let hex = rgbToHex(parseInt(rgb[0]), parseInt(rgb[1]), parseInt(rgb[2]));
            if (useTransparencyCheckbox.checked) {
                const alpha = Math.round(parseInt(transparencyInput.value) * 2.55).toString(16).padStart(2, '0');
                hex += alpha;
            }
            const position = parseFloat(stop.style.left);
            return `${hex} ${position.toFixed(2)}%`;
        });

        const direction = getGradientDirection();
        const gradient = `linear-gradient(${direction}, ${gradientStops.join(', ')})`;
        gradientPreview.style.background = gradient;
    }

    function getGradientDirection() {
        const selectedDirection = document.querySelector('input[name="gradientDirection"]:checked').value;
        switch (selectedDirection) {
            case 'to right': return '90deg';
            case 'to bottom right': return '135deg';
            case 'to bottom': return '180deg';
            case 'to bottom left': return '225deg';
            case 'to left': return '270deg';
            case 'to top left': return '315deg';
            case 'to top': return '0deg';
            case 'to top right': return '45deg';
            case 'custom': return `${customAngleInput.value}deg`;
            default: return selectedDirection;
        }
    }

    function rgbToHex(r, g, b) {
        return "#" + [r, g, b].map(x => x.toString(16).padStart(2, '0')).join('');
    }

    function exportGradientCSS() {
        const stops = document.querySelectorAll('.gradientStop');
        const gradientStops = Array.from(stops).map(stop => {
            const rgb = stop.style.backgroundColor.match(/\d+/g);
            let hex = rgbToHex(parseInt(rgb[0]), parseInt(rgb[1]), parseInt(rgb[2]));
            if (useTransparencyCheckbox.checked) {
                const alpha = Math.round(parseInt(transparencyInput.value) * 2.55).toString(16).padStart(2, '0');
                hex += alpha;
            }
            const position = parseFloat(stop.style.left);
            return `${hex} ${position.toFixed(2)}%`;
        });

        const direction = getGradientDirection();
        const gradientCSS = `background: linear-gradient(${direction}, ${gradientStops.join(', ')});`;

        navigator.clipboard.writeText(gradientCSS).then(() => {
            const message = document.getElementById('copyMessage');
            message.style.display = 'block';
            setTimeout(() => {
                message.style.display = 'none';
            }, 6000);
        });
    }




    // Neue Import-Funktion
    function importGradient() {
    const cssText = importText.value;
    const gradientRegex = /linear-gradient\((.*?)\)/;
    const match = cssText.match(gradientRegex);

    if (match) {
        const gradientContent = match[1];
        const colorStopRegex = /(#[0-9A-Fa-f]{3,8}|rgba?\([\d\s,\.]+\))\s*([\d\.]+%)?/g;
        const colorStops = [];
        let colorStopMatch;

        while ((colorStopMatch = colorStopRegex.exec(gradientContent)) !== null) {
            colorStops.push({
                color: colorStopMatch[1],
                position: colorStopMatch[2] ? parseFloat(colorStopMatch[2]) : null
            });
        }

        initGradientEditor(colorStops);
    } else {
        alert('Ungültiger CSS-Gradient-Code. Bitte überprüfen Sie den eingegebenen Text.');
    }
}

    updateGradientBtn.addEventListener('click', () => initGradientEditor());
    exportGradientCSSBtn.addEventListener('click', exportGradientCSS);
    document.querySelectorAll('input[name="gradientDirection"]').forEach(radio => {
        radio.addEventListener('change', updateGradientPreview);
    });
    customAngleInput.addEventListener('input', updateGradientPreview);
    useTransparencyCheckbox.addEventListener('change', updateGradientPreview);
    transparencyInput.addEventListener('input', updateGradientPreview);
    colorPickerModeCheckbox.addEventListener('change', (e) => {
        colorPickerMode = e.target.checked;
    });

    // Neuer Event Listener für den Import-Button
    importButton.addEventListener('click', importGradient);

    // Initialisierung
    initGradientEditor();
});
document.addEventListener('DOMContentLoaded', function() {
  const toggleImportBtn = document.getElementById('toggleImportBtn');
  const importSection = document.getElementById('importSection');

  toggleImportBtn.addEventListener('click', function() {
    if (importSection.style.display === 'none') {
      importSection.style.display = 'block';
    } else {
      importSection.style.display = 'none';
    }
  });

  // Hier Ihren bestehenden Import-Code einfügen
});

const customAngleInput = document.getElementById('customAngle');
const customArrow = document.querySelector('.direction-arrow.custom');

function rotateArrow(angle) {
  // Subtrahieren Sie 90 von dem Winkel, um die korrekte Ausrichtung zu erhalten
  const adjustedAngle = angle - 90;
  customArrow.style.transform = `rotate(${adjustedAngle}deg)`;
}

customAngleInput.addEventListener('input', function() {
  rotateArrow(this.value);
});

document.querySelector('input[name="gradientDirection"][value="custom"]').addEventListener('change', function() {
  if (this.checked) {
    rotateArrow(customAngleInput.value);
  }
});

<!-- Accordion -->
function setupAccordion(accordionId) {
        const accordion = document.getElementById(accordionId);
        accordion.addEventListener('click', function(e) {
            if (e.target.classList.contains('accordion_title')) {
                const content = e.target.nextElementSibling;
                const isActive = e.target.classList.contains('active');
                
                this.querySelectorAll('.accordion_title').forEach(title => {
                    title.classList.remove('active');
                    title.nextElementSibling.classList.remove('active');
                });

                if (!isActive) {
                    e.target.classList.add('active');
                    content.classList.add('active');
                }
            }
        });
    }

    setupAccordion('accordion_1');
	
	function resetPage() {
    location.reload();
}

    </script>
	
	<div id="notification"></div>

<footer class="footer_main">
    <div class="footer_content">
        <div class="footer_copyright">
            <span class="copyright_text">Copyright by</span> 
            <a class="copyright_url" href="https://webtools.whitefox-webdesign.de" target="_blank" rel="noopener">
                Whitefox-Webdesign.de
            </a> 
            <span class="copyright_year">2018 - <?php echo date("Y"); ?></span>
        </div>
    </div>
</footer>

</body>
</html>


